import { Action, ActionPanel, Clipboard, Form, List, Toast, showToast, useNavigation } from "@raycast/api";
import { useEffect, useMemo, useState } from "react";
import { PromptDocument, PromptIndexEntry, listPrompts, runPrompt, showPrompt } from "./lib/spore";

export default function Command() {
  const [isLoading, setIsLoading] = useState(true);
  const [items, setItems] = useState<PromptIndexEntry[]>([]);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    async function load() {
      try {
        const index = await listPrompts();
        setItems(index.prompts || []);
      } catch (err) {
        setError(err instanceof Error ? err.message : String(err));
      } finally {
        setIsLoading(false);
      }
    }
    load();
  }, []);

  return (
    <List isLoading={isLoading} searchBarPlaceholder="Search prompts...">
      {error ? <List.EmptyView title="Failed to load prompts" description={error} /> : null}
      {items.map((item) => (
        <List.Item
          key={item.id}
          title={item.title}
          subtitle={item.goal || undefined}
          accessories={item.tags?.length ? [{ tag: item.tags.join(", ") }] : []}
          actions={<PromptActions promptId={item.id} />}
        />
      ))}
    </List>
  );
}

function PromptActions({ promptId }: { promptId: string }) {
  const { push } = useNavigation();

  return (
    <ActionPanel>
      <Action title="Run Prompt" onAction={() => push(<RunForm promptId={promptId} />)} />
    </ActionPanel>
  );
}

function RunForm({ promptId }: { promptId: string }) {
  const [prompt, setPrompt] = useState<PromptDocument | null>(null);
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    async function load() {
      try {
        const data = await showPrompt(promptId);
        setPrompt(data);
      } finally {
        setIsLoading(false);
      }
    }
    load();
  }, [promptId]);

  const initialValues = useMemo(() => {
    if (!prompt) return {};
    const values: Record<string, string | boolean> = {};
    for (const [key, input] of Object.entries(prompt.inputs || {})) {
      if (input.type === "boolean") {
        values[key] = Boolean(input.default ?? false);
      } else if (input.default !== undefined && input.default !== null) {
        values[key] = String(input.default);
      } else {
        values[key] = "";
      }
    }
    return values;
  }, [prompt]);

  async function handleSubmit(values: Record<string, string | boolean>) {
    if (!prompt) return;
    const inputs: Record<string, string> = {};
    for (const [key, value] of Object.entries(values)) {
      if (typeof value === "boolean") {
        inputs[key] = value ? "true" : "false";
      } else {
        inputs[key] = value;
      }
    }

    await showToast({ style: Toast.Style.Animated, title: "Running prompt..." });
    try {
      const result = await runPrompt(prompt.id, inputs);
      await Clipboard.copy(result.output);
      await showToast({ style: Toast.Style.Success, title: "Output copied to clipboard" });
    } catch (error) {
      const err = error as { stderr?: string; message?: string };
      await showToast({
        style: Toast.Style.Failure,
        title: "Failed to run prompt",
        message: err.stderr || err.message || String(error),
      });
    }
  }

  if (isLoading || !prompt) {
    return <Form isLoading={true} />;
  }

  return (
    <Form
      navigationTitle={prompt.title}
      actions={
        <ActionPanel>
          <Action.SubmitForm title="Run" onSubmit={handleSubmit} />
        </ActionPanel>
      }
    >
      {Object.entries(prompt.inputs || {}).map(([key, input]) => {
        const defaultValue = initialValues[key];

        if (input.type === "enum" && input.enum?.length) {
          return (
            <Form.Dropdown id={key} key={key} title={key} defaultValue={String(defaultValue)}>
              {input.enum.map((value) => (
                <Form.Dropdown.Item key={value} value={value} title={value} />
              ))}
            </Form.Dropdown>
          );
        }

        if (input.type === "boolean") {
          return (
            <Form.Checkbox
              id={key}
              key={key}
              title={key}
              label={input.description || key}
              defaultValue={defaultValue as boolean}
            />
          );
        }

        if (input.type === "text") {
          return (
            <Form.TextArea
              id={key}
              key={key}
              title={key}
              placeholder={input.description || ""}
              defaultValue={defaultValue as string}
            />
          );
        }

        return (
          <Form.TextField
            id={key}
            key={key}
            title={key}
            placeholder={input.description || ""}
            defaultValue={defaultValue as string}
          />
        );
      })}
    </Form>
  );
}
