import { getPreferenceValues } from "@raycast/api";
import { execa } from "execa";
import { homedir } from "node:os";
import { join } from "node:path";

export type PromptIndex = {
  version: number;
  generated_at?: string;
  generatedAt?: string;
  prompts: PromptIndexEntry[];
};

export type PromptIndexEntry = {
  id: string;
  pulse_mode?: string;
  pulseMode?: string;
  pulse_vibe?: string;
  pulseVibe?: string;
  confidentiality?: "low" | "medium" | "high";
  tribe_id?: string | null;
  tribeId?: string | null;
  title: string;
  goal?: string | null;
  tags: string[];
  path: string;
  updated_at?: string;
  updatedAt?: string;
  has_tests?: boolean;
  hasTests?: boolean;
  inputs: string[];
};

export type InputDefinition = {
  type: "string" | "text" | "number" | "boolean" | "enum" | "json";
  required: boolean;
  default?: string | number | boolean | null;
  enum?: string[];
  pattern?: string;
  description?: string;
};

export type PromptDocument = {
  id: string;
  promptfile?: number;
  pulse_mode?: string;
  pulseMode?: string;
  pulse_vibe?: string;
  pulseVibe?: string;
  tribe_id?: string | null;
  tribeId?: string | null;
  confidentiality?: "low" | "medium" | "high";
  version?: string | number;
  parent_hash?: string | null;
  parentHash?: string | null;
  title: string;
  goal?: string | null;
  tags: string[];
  inputs: Record<string, InputDefinition>;
  output?: { format?: string; max_words?: number; max_chars?: number };
  tests?: unknown[];
  body: string;
  updated_at?: string;
  filePath: string;
};

export type RunResult = {
  prompt_id: string;
  ok: boolean;
  output: string;
  provider: string;
  model: string;
  duration_ms: number;
};

type Preferences = {
  sporePath?: string;
  workspacePath?: string;
  openAIKey?: string;
  openAIModel?: string;
  openAIBaseURL?: string;
};

function buildSwiftBuildCandidates(root?: string): string[] {
  if (!root) {
    return [];
  }

  return [
    join(root, ".build", "arm64-apple-macosx", "release", "spore"),
    join(root, ".build", "arm64-apple-macosx", "debug", "spore"),
    join(root, ".build", "x86_64-apple-macosx", "release", "spore"),
    join(root, ".build", "x86_64-apple-macosx", "debug", "spore"),
  ];
}

function uniqueCommands(commands: Array<string | undefined>): string[] {
  const seen = new Set<string>();
  const deduped: string[] = [];

  for (const command of commands) {
    const value = command?.trim();
    if (!value || seen.has(value)) {
      continue;
    }
    seen.add(value);
    deduped.push(value);
  }

  return deduped;
}

function getSporeContext() {
  const prefs = getPreferenceValues<Preferences>();
  const cwd = prefs.workspacePath?.trim() || undefined;
  const homeApps = join(homedir(), "Applications", "Spore.app", "Contents", "MacOS", "spore");
  const candidates = uniqueCommands([
    prefs.sporePath,
    "spore",
    "/opt/homebrew/bin/spore",
    "/usr/local/bin/spore",
    "/Applications/Spore.app/Contents/MacOS/spore",
    homeApps,
    ...buildSwiftBuildCandidates(cwd),
  ]);

  return { candidates, cwd, prefs };
}

async function runSpore(args: string[]) {
  const { candidates, cwd, prefs } = getSporeContext();
  const env: NodeJS.ProcessEnv = { ...process.env };
  const key = prefs.openAIKey?.trim();
  const model = prefs.openAIModel?.trim();
  const baseURL = prefs.openAIBaseURL?.trim();
  const workspacePath = prefs.workspacePath?.trim();

  if (key) {
    env.SPORE_OPENAI_API_KEY = key;
  }
  if (model) {
    env.SPORE_OPENAI_MODEL = model;
  }
  if (baseURL) {
    env.SPORE_OPENAI_BASE_URL = baseURL;
  }
  if (workspacePath) {
    env.SPORE_WORKSPACE = workspacePath;
  }

  for (const command of candidates) {
    try {
      const result = await execa(command, args, { cwd, env });
      return result.stdout;
    } catch (error) {
      const err = error as { code?: string; cause?: { code?: string } };
      const code = err?.code ?? err?.cause?.code;
      if (code === "ENOENT") {
        continue;
      }
      throw error;
    }
  }

  throw new Error(
    "Spore CLI not found. Set 'Spore CLI Path' in Raycast preferences, add `spore` to PATH, or use `/Applications/Spore.app/Contents/MacOS/spore`."
  );
}

export async function listPrompts(): Promise<PromptIndex> {
  const stdout = await runSpore(["list", "--json"]);
  return JSON.parse(stdout) as PromptIndex;
}

export async function showPrompt(id: string): Promise<PromptDocument> {
  const stdout = await runSpore(["show", id, "--json"]);
  return JSON.parse(stdout) as PromptDocument;
}

export async function runPrompt(id: string, inputs: Record<string, string>): Promise<RunResult> {
  const args = ["run", id, "--json"];
  const inputEntries = Object.entries(inputs);
  if (inputEntries.length > 0) {
    args.push("--inputs");
    for (const [key, value] of inputEntries) {
      args.push(`${key}=${value}`);
    }
  }
  const stdout = await runSpore(args);
  return JSON.parse(stdout) as RunResult;
}

export async function reindexWorkspace(): Promise<void> {
  await runSpore(["reindex"]);
}
